package no.statkart.matrikkel.eksempel;


import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.*;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.adresse.*;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.*;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.koder.BruksenhetstypeKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.koder.BygningsendringsKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.koder.BygningsstatusKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.koder.BygningstypeKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.forretning.ArealIForretning;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.forretning.ForretningIdList;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.forretning.MatrikkelenhetForretning;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.geometri.koder.KoordinatsystemKodeId;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.jobb.Jobb;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.jobb.JobbId;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.jobb.koder.JobbStatusKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.kommune.Kommune;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.kommune.KommuneId;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.kommune.KommuneIdent;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.matrikkelenhet.*;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.matrikkelenhet.koder.EierforholdKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.matrikkelenhet.koder.ForretningstypeKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.matrikkelenhet.koder.RolleIForretningKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.matrikkelenhet.koder.TinglysingsstatusKode;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.person.Person;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.rapport.MatrikkelRapport;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.rapport.OfflineMatrikkelRapport;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.rapport.koder.ExportTypeId;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.rapport.koder.MatrikkelenhetRapportTypeId;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.util.Status;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.adresse.AdresseService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.bygning.BygningService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.forretning.ForretningService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.kommune.KommuneService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.matrikkelenhet.MatrikkelenhetService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.rapport.RapportService;
import no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.StoreService;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Eksempelklient for nytt matrikkelAPI versjon 1. <br/> <br/>
 * <p/>
 * Eksempelklinten er et eksempel på navigasjon i boblemodellen som det nye APIet baserer på (se dokumentasjonen av APIet). Klienten presenterer
 * informasjon knyttet til en matrikkelenhet (i dette tilfellet en grunneiendom) utifra en oppgitt vegadresse.   <br/><br/>
 * <p/>
 * Boblemodellen innebærer domenemodellen: <br/>
 * Et domeneobjekt tilsvarer en typisk "ting" i matrikkelsystemet (matrikkelenhet, adresse ol.), samt regler for denne tingen. <br/>
 * En samling assosiserte domeneobjekter samles i en boble som tilsvarer et domene. Objektene innad i en boble refererer til hverandre (pekere). <br/>
 * En boble kan aksesseres gjennom en global id som tilsvarer id'en for rotobjektet som "eier" boblen
 * (f.eks. vil en matrikkelenhet ha en id som refererer til hele matrikkelenhetboblen).<br/>
 * Alle bobleobjekter er subklasser av klassen <code>MatrikkelBubbleObject</code>. <br/>
 * Alle bobleId'er er subklasser av klassen <code>MatrikkelBubbleId</code>. <br/><br/>
 * <p/>
 * Følgelig vil en algoritme for aksess på data i matrikkelen være noe ala:<br/>
 * 1. En logisk identifikator er oppgitt. (Her: Kommunenummer er oppgitt, det er tilstrekkelig med opplysninger for å kunne identifisere en kommune logisk.)<br/>
 * 2. Finner id til en boble utifra den logiske identifikatoren vha. boblens tilhørende tjeneste. (Her: Finner en KommuneId med en metode som tilhører KommuneService utifra Kommunenummer)<br/>
 * 3. Henter rotobjektet for boblen gjennom tjenesten StoreService utifra id til boblen. Dette rotobjektet har metoder knyttet til seg for navigasjon til opplysninger som
 * boblen omfatter. (Her: Finner en kommune som inneholder kommunenavnet).  <br/>
 * StoreService operer med superklassene <code>MatrikkelBubbleObject</code> og <code>MatrikkelBubbleId</code>. Navigasjon i bobler krever derfor casting.<br/><br/>
 * <p/>
 * <p/>
 * Eksempelklienten har flere eksempler på denne algoritmen: <br/>
 * En vegadresse er oppgitt.<br/>
 * Utifra vegadressen hentes id til én kommuneboble, id til én matrikkelenhetboble og id til én adresseboble. <br/>
 * Henter opplysninger vi ønsker å displaye for oppgitt adresse ved å navigere i boblene over. <br/>.
 * Navigerer oss også inn i andre bobler utifra boblene over, f.eks. finner vi opplysninger om teiger ved å gå fra MatrikkelenhetId til TeigId.
 * Til slutt genereres en rapport gjennom RapportService. <br/>
 * <p/>
 * <p/>
 * Tankegangen i implementasjonen av eksempelklienten er forsøkt dokumentert.
 */

public class Eksempelklient {
    /**
     * Konfigureres via
     * {@code -Dmatrikkel_server="https://www.test.matrikkel.no:443" -Dmatrikkel_username=brukernavn -Dmatrikkel_password=xxx }
     *
     */
    public static void main(String[] args) throws no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException {
        final String username = System.getProperty("matrikkel_username");
        final String password = System.getProperty("matrikkel_password");
        final String address = System.getProperty("matrikkel_server");

        if (username == null) {
            System.err.println("Mangler system property for username");
            System.exit(-1);
        }
        if (password == null) {
            System.err.println("Mangler system property for password");
            System.exit(-1);
        }
        if (address == null) {
            System.err.println("Mangler system property for host");
            System.exit(-1);
        }

        ServiceFactory serviceFactory = new ServiceFactory(username, password, address);

        //Oppretter nødvendige servicer:
        AdresseService adresseService = serviceFactory.adresseService();
        MatrikkelenhetService matrikkelenhetService = serviceFactory.matrikkelenhetService();
        StoreService storeService = serviceFactory.storeService();
        KommuneService kommuneService = serviceFactory.kommuneService();
        BygningService bygningService = serviceFactory.bygningService();
        ForretningService forretningService = serviceFactory.forretningService();
        RapportService rapportService = serviceFactory.rapportService();


        //Oppretter en context for metodene i tjenestene (som gir oss data) se https://www.test.matrikkel.no/matrikkelapi/wsapi/v1/dokumentasjon/matrikkelAPI.pdf for forklaring på de forskjellige feltene:
        MatrikkelContext matrikkelContext = new MatrikkelContext();
        matrikkelContext.setLocale("no_NO");
        matrikkelContext.setBrukOriginaleKoordinater(true);
        KoordinatsystemKodeId koordinatsystemKodeId = new KoordinatsystemKodeId();
        koordinatsystemKodeId.setValue(10);
        matrikkelContext.setKoordinatsystemKodeId(koordinatsystemKodeId);
        matrikkelContext.setSystemVersion("3.17"); //Dette angir hvilken versjon av api-et man støtter.
        matrikkelContext.setKlientIdentifikasjon("Min testklient"); //Denne må være satt til en gyldig klientidentifikasjon for oppdatering for å kunne bestille rapport

        Timestamp timestamp = new Timestamp();
        try {
            //9999-01-01 00:00:00:00+01:00 gir data fra sanntid, nyeste versjon. Et annet tidspunkt returnerer data som eksisterte på det tidspunktet.
            //OBS! timezone er i antall minutter ikke antall timer
            XMLGregorianCalendar currentTime = DatatypeFactory.newInstance().newXMLGregorianCalendar(9999, 1, 1, 0, 0, 0, 0, 60);
            timestamp.setTimestamp(currentTime);
        } catch (DatatypeConfigurationException e) {
            System.out.println(e.getMessage());
            System.exit(-1);
        }
        matrikkelContext.setSnapshotVersion(timestamp);


        //Konfigurasjon ferdig, begynner navigasjon i domenemodellen.
        System.out.println("\nDENNE EKSEMPELKLIENTEN FINNER OBJEKTINFORMASJON OM EN MATRIKKELENHET (I DETTE TILFELLET EN GRUNNEIENDDOM) UTIFRA EN VEGADRESSE: ");

        //En vegadresse er oppgitt, den gir opphav til logiske identifikatorer for objekter i matrikkelen
        String kommunenummer = "3305"; //Kommunenummer er logisk identifikator for en kommune
        String adressenavn = "Høybyveien";
        Integer husNummer = 11;
        String bokstav = "A";


        //Finner tilhørende kommunenavn etter generell algoritme for å aksessere databasen:
        //1. Oppretter først en "ident", dvs. en logisk identifikator, for å finne et kommuneobjekt utifra oppgitte data.
        KommuneIdent kommuneident = new KommuneIdent();
        kommuneident.setKommunenummer(kommunenummer);
        KommuneId kommuneId = null;   //Den globale kommuneid'en i eksempelklienten.
        try {
            //2. Finner bobleid vha. relevant tjeneste for boble.
            kommuneId = kommuneService.findKommuneIdForIdent(kommuneident, matrikkelContext);
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.kommune.ServiceException e) { //Hver tjeneste har egen ServiceException-klasse
            System.err.println("Feil, finner ikke kommunid: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        Kommune kommune = null;
        try {
            //3. Henter rotobjekt med tjenesten StoreService for å nå ønsket data (kommuneobjektet brukes senere for å hente ut navn på kommune).
            kommune = (Kommune) storeService.getObject(kommuneId, matrikkelContext); //Må caste til rett bobleobjekt
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
            System.err.println("Feil, finner ikke kommuneobjekt: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }

        System.out.println("Oppgitt vegadresse: " + adressenavn + " " + husNummer + bokstav + " i kommunen " + kommune.getKommunenavn() + "\n");


        //Man kan også finne en bobleid gjennom en søkemodell
        //Genererer en søkemodell for adresse for å finne id for matrikkelenhetboble
        AdressesokModel adressesokModel = new AdressesokModel();
        adressesokModel.setAdressetype(Adressetype.VEGADRESSE);
        adressesokModel.setAdressenavn(adressenavn);
        adressesokModel.setKommunenummer(kommunenummer);
        adressesokModel.setBokstav(bokstav);
        adressesokModel.setHusnummer(husNummer.toString());

        //Finner id for bobler: Først for adresseboble, deretter for matrikkelenhetboble.
        AdresseId adresseId = null;
        MatrikkelenhetId matrikkelenhetId = null;
        try {
            AdresseIdList adresseIds = adresseService.findAdresser(adressesokModel, matrikkelContext);
            adresseId = adresseIds.getItem().get(0);     //Velger mest relevante resultat
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.adresse.ServiceException e) {
            System.err.println("Feil, finner ikke adresseId for søkemodell: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        try {
            MatrikkelenhetIdList matrikkelenhetIdList = matrikkelenhetService.findMatrikkelenheterForAdresse(adresseId, false, true, matrikkelContext);
            matrikkelenhetId = matrikkelenhetIdList.getItem().get(0); //Velger mest relevante resultat
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.matrikkelenhet.ServiceException e) {
            System.err.println("Feil, finner ikke matrikkelId for adresse: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }

        //Finner rotobjekt i adresseboble og matrikkelenhetboble
        Matrikkelenhet matrikkelenhet = null;
        Vegadresse vegadresse = null;
        try {
            vegadresse = (Vegadresse) storeService.getObject(adresseId, matrikkelContext);
            matrikkelenhet = (Matrikkelenhet) storeService.getObject(matrikkelenhetId, matrikkelContext);
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
            System.err.println("Feil, finner ikke bobleobjekter, adresse, matrikkelenhet: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }

        //Henter ønskede opplysninger med metoder knyttet til bobleobjekt.
        //Navigerer kun i matrikkelenhetboblen, ikke i adresseboblen.
        Matrikkelnummer matrikkelnummer = matrikkelenhet.getMatrikkelnummer();
        String matrikkelnummerString = matrikkelnummer.getGardsnummer() + "/" + matrikkelnummer.getBruksnummer() + "/" + matrikkelnummer.getFestenummer() + "/" + matrikkelnummer.getSeksjonsnummer();
        String bruksnavn = matrikkelenhet.getBruksnavn();
        LocalDate etableringsDato = matrikkelenhet.getEtableringsdato();
        boolean harKulturminne = matrikkelenhet.isHarKulturminne();
        String matrikkelenhetType = matrikkelenhet.getClass().getSimpleName();
        boolean harAktiveFestegrunner = matrikkelenhet.isHarAktiveFestegrunner();
        boolean harTinglyst = matrikkelenhet.isTinglyst();

        //Navigerer fra et bobleobjekt til et annet, dvs. får tak i en annen boble sin id.
        TeigForMatrikkelenhetList teigForMatrikkelenhetList = matrikkelenhet.getTeigerForMatrikkelenhet();
        Integer antTeiger = teigForMatrikkelenhetList.getItem().size();
        Double areal = 0.0;
        //Har flere teiger for en matrikkelenhet, derfor loop
        for (Integer i = 0; i < antTeiger; i++) {
            TeigId teigId = teigForMatrikkelenhetList.getItem().get(i).getTeigId();
            Teig teig = null;
            try {
                teig = (Teig) storeService.getObject(teigId, matrikkelContext);    //Henter bobleobjekt
            } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
                System.out.println("Feil, finner ikke teig: " + e.getMessage());
                e.printStackTrace();
                System.exit(-2);
            }
            areal += teig.getLagretBeregnetAreal();
        }

        //Printer opplysninger
        System.out.println("AdresseId: " + adresseId.getValue());
        System.out.println("MatrikkelenhetId: " + matrikkelenhetId.getValue());
        System.out.println("Matrikkelnummer: " + matrikkelnummerString);
        System.out.println("Type: " + matrikkelenhetType);
        System.out.println("Bruksnavn: " + bruksnavn);
        System.out.println("Etableringsdato: " + etableringsDato.getDate());
        System.out.println("Tinglyst?: " + harTinglyst);
        System.out.println("Har kulturminne?: " + harKulturminne);
        System.out.println("Aktive festegrunner?: " + harAktiveFestegrunner);
        System.out.println("Antall teiger: " + antTeiger);
        System.out.println("Areal: " + areal);


        //Finner tilknyttede bygninger for matrikkelenhet:
        ByggIdList byggIdList = null;
        try {
            //Finner bobleid
            byggIdList = bygningService.findByggForMatrikkelenhet(matrikkelenhetId, matrikkelContext);
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.bygning.ServiceException e) {
            System.err.println("Feil, finner ikke byggId: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        List<Bygg> byggList = new ArrayList<>();
        try {
            /*
            getObjects tar argument av typen matrikkelBubbleIdList, har ingen barn, må derfor gå veien innom MatrikkelBubbleId for å få sendt inn innhold i byggIdList til storeService:
            For å hente ut flere rotobjekter samtidig, benyttes typen MatrikkelBubbleIdList på arugmentene til StoreService.
            Implementasjonen av boblemodellen innebærer arv av id-klasser, ikke list-klasser.
            List-klassene inneholder et attributt av typen List som inneholder elementer av en spesifikk id. (Eksempelvis: List<...Id>)
            Går veien om dette attributtet for å få rett type på id'ene som skal sendes inn, da tjenesten StoreService krever argument av
            typen MatrikkelBubbleId (superklasse for alle id-klasser).
             */
            MatrikkelBubbleIdList matrikkelBubbleIdListBygg = new MatrikkelBubbleIdList();
            List<MatrikkelBubbleId> refItem = matrikkelBubbleIdListBygg.getItem();
            refItem.addAll(byggIdList.getItem());

            //Finner bobleobjekt
            MatrikkelBubbleObjectList matrikkelBubbleObjectListBygg = storeService.getObjects(matrikkelBubbleIdListBygg, matrikkelContext);
            //Caster elementer til lista til objekter av typen Bygg
            byggList.addAll((List<Bygg>)(Collection<?>) matrikkelBubbleObjectListBygg.getItem());

        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
            System.err.println("Feil, finner ikke liste av bygg: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        System.out.println("\nAntall bygg (bygningsendring eller en bygning) knyttet til matrikkelenhet: " + byggList.size() + "\n");


        //Iterer gjennom alle bygg
        for (int i = 0; i < byggList.size(); ) {
            Bygg bygg = byggList.get(i);
            System.out.println((++i) + ". " + bygg.getClass().getSimpleName());

            if (bygg instanceof Bygning) {
                Bygning bygning = (Bygning) bygg;
                //Kodelister utgjør også egne bobler, men arver ikke direkte fra MatrikkelBubbleObject/Id som domenebojektene gjør.(Går via klassen <code>Kode</code>/<code>KodeId</code> med mer.)
                BygningstypeKode bygningstype = (BygningstypeKode) storeService.getObject(bygning.getBygningstypeKodeId(), matrikkelContext);     //kan kaste ServiceException, unhandled her
                System.out.println("\tType: " + bygningstype.getNavn().getEntry().get(0).getValue());
            } else if (bygg instanceof Bygningsendring) {
                Bygningsendring bygningsendring = (Bygningsendring) bygg;
                BygningsendringsKode bygningsendringsKode = (BygningsendringsKode) storeService.getObject(bygningsendring.getBygningsendringsKodeId(), matrikkelContext);  //kan kaste ServiceException, unhandled her
                System.out.println("\tType: " + bygningsendringsKode.getNavn().getEntry().get(0).getValue());
            }

            //Statushistorikk:
            System.out.println("\tStatus: ");
            List<BygningsstatusHistorikk> BygningsstatusHistorikkList = bygg.getBygningsstatusHistorikker().getItem();
            for (no.statkart.matrikkel.matrikkelapi.wsapi.v1.domain.bygning.BygningsstatusHistorikk BygningsstatusHistorikk : BygningsstatusHistorikkList) {
                //Kodeliste krever at vi går via id og storeService
                BygningsstatusKode BygningsstatusKode = (BygningsstatusKode) storeService.getObject(BygningsstatusHistorikk.getBygningsstatusKodeId(), matrikkelContext); //kan kaste ServiceException, unhandled her

                LocalDate date = BygningsstatusHistorikk.getDato();
                String dateString;
                if (date != null) {
                    dateString = date.getDate().toString();
                } else {
                    dateString = " ";
                }
                System.out.println("\t\t" + BygningsstatusKode.getNavn().getEntry().get(0).getValue() + " " + dateString);
            }

            System.out.print("\tBruksenheter til bygg: ");
            BruksenhetIdList bruksenhetIdList = bygg.getBruksenhetIds();
            List<Bruksenhet> bruksenhetList = new ArrayList<>();
            try {
                MatrikkelBubbleIdList matrikkelBubbleIdListBruksenhet = new MatrikkelBubbleIdList();
                List<MatrikkelBubbleId> refItem = matrikkelBubbleIdListBruksenhet.getItem();
                refItem.addAll(bruksenhetIdList.getItem());

                MatrikkelBubbleObjectList matrikkelBubbleObjectListBruksenhet = storeService.getObjects(matrikkelBubbleIdListBruksenhet, matrikkelContext);
                //Caster elementer til lista til objekter av typen Bruksenhet
                bruksenhetList.addAll((List<Bruksenhet>) (List<?>) matrikkelBubbleObjectListBruksenhet.getItem());
            } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
                System.err.println("Feil, finner ikke bruksenheter: " + e.getMessage());
                e.printStackTrace();
                System.exit(-2);
            }
            for (Bruksenhet bruksenhet : bruksenhetList) {
                BruksenhetstypeKode bruksenhetstype = (BruksenhetstypeKode) storeService.getObject(bruksenhet.getBruksenhetstypeKodeId(), matrikkelContext);  //kan kaste ServiceException, unhandled her
                System.out.print(bruksenhetstype.getNavn().getEntry().get(0).getValue() + " ");
            }
            System.out.println(" ");

        }


        //Eierforhold
        System.out.println("\nEierforhold:\n");
        //Navigerer i matrikkelenhet-boble
        EierforholdList eierforholdList = matrikkelenhet.getEierforhold();
        for (int i = 0; i < eierforholdList.getItem().size(); i++) {
            Eierforhold eierforhold = eierforholdList.getItem().get(i);
            //Kodeliste krever at vi går via id og storeService
            EierforholdKode eierforholdKode = (EierforholdKode) storeService.getObject(eierforhold.getEierforholdKodeId(), matrikkelContext);   //kan kaste ServiceException, unhandled her
            System.out.print(eierforhold.getClass().getSimpleName() + ": " + eierforholdKode.getNavn().getEntry().get(0).getValue());

            //Eierforhold har 3 ulike subklasser, må caste rett for å kunne anvende de rette metodene for å aksessere ønsket data
            if (eierforhold instanceof IkkeTinglystEierforhold) {
                IkkeTinglystEierforhold ikkeTinglystEierforhold = (IkkeTinglystEierforhold) eierforhold;
                //Henter eierandel
                System.out.print("(" + ikkeTinglystEierforhold.getAndel().getTeller() + "/" + ikkeTinglystEierforhold.getAndel().getNevner() + "), ");
                System.out.print("dato fra: " + ikkeTinglystEierforhold.getDatoFra().getDate() + ", ");
                if (ikkeTinglystEierforhold instanceof PersonIkkeTinglystEierforhold) {
                    //Henter navn
                    PersonIkkeTinglystEierforhold personIkkeTinglyst = (PersonIkkeTinglystEierforhold) ikkeTinglystEierforhold;
                    Person person = (Person) storeService.getObject(personIkkeTinglyst.getEierId(), matrikkelContext);            //kan kaste ServiceException, unhandled her
                    System.out.println(person.getNavn() + "\n");
                }
            } else if (eierforhold instanceof TinglystEierforhold) {
                TinglystEierforhold tinglystEierforhold = (TinglystEierforhold) eierforhold;
                //Henter eierandel
                System.out.print("(" + tinglystEierforhold.getAndel().getTeller() + "/" + tinglystEierforhold.getAndel().getNevner() + "), ");
                System.out.print("dato fra: " + tinglystEierforhold.getDatoFra().getDate() + ", ");
                if (tinglystEierforhold instanceof PersonTinglystEierforhold) {
                    //Henter navn
                    PersonTinglystEierforhold personTinglyst = (PersonTinglystEierforhold) tinglystEierforhold;
                    Person person = (Person) storeService.getObject(personTinglyst.getEierId(), matrikkelContext);  //kan kaste ServiceException, unhandled her
                    System.out.println(person.getNavn());
                }
            } else if (eierforhold instanceof Kontaktinstans) {
                //Henter kun navn
                Kontaktinstans kontaktinstans = (Kontaktinstans) eierforhold;
                Person person = (Person) storeService.getObject(kontaktinstans.getEierId(), matrikkelContext);  //kan kaste ServiceException, unhandled her
                System.out.println(", dato fra: " + kontaktinstans.getDatoFra().getDate() + person.getNavn());
            }
        }

        //Finner forretninger
        System.out.println("\nForretninger:\n");
        ForretningIdList forretningIdList = null;
        //Henter id for forretningsboble
        try {
            forretningIdList = forretningService.findForretningerForMatrikkelenhet(matrikkelenhetId, matrikkelContext);
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.forretning.ServiceException e) {
            System.err.println("Feil, finner ikke id av forretninger: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        List<MatrikkelenhetForretning> forretningList = new ArrayList<>();
        //Henter rotobjektet Forretning i forretningsboblen.
        //Har fått returnert en List-klasse. Omvei om List-attributt for å få rett type på argument til StoreService
        try {
            MatrikkelBubbleIdList matrikkelBubbleIdListForretning = new MatrikkelBubbleIdList();
            List<MatrikkelBubbleId> refItem = matrikkelBubbleIdListForretning.getItem();
            refItem.addAll(forretningIdList.getItem());

            MatrikkelBubbleObjectList matrikkelBubbleObjectListForretning = storeService.getObjects(matrikkelBubbleIdListForretning, matrikkelContext);
            forretningList.addAll((List<MatrikkelenhetForretning>) (List<?>) matrikkelBubbleObjectListForretning.getItem());
        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.store.ServiceException e) {
            System.err.println("Feil, finner ikke forretninger: " + e.getMessage());
            e.printStackTrace();
            System.exit(-2);
        }
        for (int i = 0; i < forretningList.size(); ) {
            MatrikkelenhetForretning forretning = forretningList.get(i);
            //Kodeliste krever at vi går via id og storeService
            ForretningstypeKode forretningsType = (ForretningstypeKode) storeService.getObject(forretning.getForretningstypeKodeId(), matrikkelContext); //kan kaste ServiceException, unhandled her
            System.out.println((++i) + ". " + forretningsType.getNavn().getEntry().get(0).getValue() + " " + forretning.getForretningsdokumentdato().getDate());

            //Kodeliste krever at vi går via id og storeService
            TinglysingsstatusKode tinglysingsstatus = (TinglysingsstatusKode) storeService.getObject(forretning.getTinglysingsstatusKodeId(), matrikkelContext);  //kan kaste ServiceException, unhandled her
            System.out.println("\tTinglysingsstatus: " + tinglysingsstatus.getNavn().getEntry().get(0).getValue());

            //Henter ut berørte matrikkelenheter i forretningen
            System.out.println("\tInvolverte: ");
            List<ArealIForretning> arealIForretningList = forretning.getArealIForretninger().getItem();
            for (ArealIForretning arealIForretning : arealIForretningList) {
                //Kodeliste krever at vi går via id og storeService
                RolleIForretningKode rolleIForretning = (RolleIForretningKode) storeService.getObject(arealIForretning.getRolleId(), matrikkelContext);   //kan kaste ServiceException, unhandled her

                //Identifiserer berørte matrikkelenheter med matrikkelnummer og kommunenummer, samt hvor mye areal det er snakk om.
                Matrikkelenhet matrikkelenhetIForretning = (Matrikkelenhet) storeService.getObject(arealIForretning.getMatrikkelenhetId(), matrikkelContext);     //kan kaste ServiceException, unhandled her
                Matrikkelnummer matrikkelnrIForretning = matrikkelenhetIForretning.getMatrikkelnummer();
                Kommune kommuneIForretning = (Kommune) storeService.getObject(matrikkelnrIForretning.getKommuneId(), matrikkelContext);                           //kan kaste ServiceException, unhandled her
                String matrikkelnrString = kommuneIForretning.getKommunenummer() + " - " + matrikkelnrIForretning.getGardsnummer() + "/" + matrikkelnrIForretning.getBruksnummer() + "/" + matrikkelnrIForretning.getFestenummer() + "/" + matrikkelnrIForretning.getSeksjonsnummer();
                System.out.println("\t\t" + matrikkelnrString + " (" + rolleIForretning.getNavn().getEntry().get(0).getValue() + ", " + arealIForretning.getArealendring() + ")");
            }
        }

/*
        TODO: Hvis man ønsker å teste bestilling av rapport kan man kommentere inn denne delen av koden. Dette krever en gyldig klientidentifikasjon

        //Genererer matrikkelenhetrapport, typen enkel
        System.out.println("\nVi forsoker å generere rapporten \"Matrikkelenhetrapport\", enkel: \n");
        boolean offlineStatus = true;
        //Oppsett av søkemodell for matrikkelenhet basert på opplysninger vi har funnet til nå
        MatrikkelenhetsokModel matrikkelenhetsokModel = new MatrikkelenhetsokModel();
        matrikkelenhetsokModel.setKommunenummer(kommunenummer);

        matrikkelenhetsokModel.setGardsnummer(matrikkelnummer.getGardsnummer().toString());
        matrikkelenhetsokModel.setBruksnummer(matrikkelnummer.getBruksnummer().toString());
        matrikkelenhetsokModel.setFestenummer(matrikkelnummer.getFestenummer());
        matrikkelenhetsokModel.setSeksjonsnummer(matrikkelnummer.getSeksjonsnummer());
        matrikkelenhetsokModel.setStatus(Status.BESTAENDE);

        MatrikkelenhetRapportTypeId rapportTypeId = new MatrikkelenhetRapportTypeId();
        rapportTypeId.setValue(1); //Rapporttypen enkel
        ExportTypeId exportTypeId = new ExportTypeId();
        exportTypeId.setValue(2); //Generer i pdf-format
        try {
            OfflineMatrikkelRapport offlineMatrikkelRapport = (OfflineMatrikkelRapport) rapportService.createMatrikkelenhetRapport(matrikkelenhetsokModel, rapportTypeId, exportTypeId, offlineStatus, matrikkelContext);
            JobbId jobbId = offlineMatrikkelRapport.getJobbId();
            Jobb jobb = (Jobb) storeService.getObject(jobbId, matrikkelContext);
            JobbStatusKode jobbstatus = (JobbStatusKode) storeService.getObject(jobb.getStatusId(), matrikkelContext);
            String jobbstatusString = jobbstatus.getNavn().getEntry().get(0).getValue();

            //Vi venter til jobben er ferdiggenerert og klar til å hentes.
            while (!jobbstatusString.equals("Klar") && !jobbstatusString.equals("Feil")) {
                System.out.println("Rapporten er ikke ferdiggenerert, vi venter i 1 min. Rapportstatus: " + jobbstatusString);
                try {
                    Thread.sleep(60000);
                } catch (InterruptedException e) {
                    System.exit(-1);
                }
                //Oppdaterer status på rapport-jobben.
                jobb = (Jobb) storeService.getObject(jobbId, matrikkelContext);
                jobbstatus = (JobbStatusKode) storeService.getObject(jobb.getStatusId(), matrikkelContext);
                jobbstatusString = jobbstatus.getNavn().getEntry().get(0).getValue();

            }
            System.out.println("Rapportstatus: " + jobbstatusString);

            //Henter rapporten, representerer den med en url som kan limes inn i nettleser for framvisning.
            MatrikkelRapport matrikkelRapport = rapportService.hentRapport(jobbId, matrikkelContext);
            System.out.println("URL til rapport: " + matrikkelRapport.getUrl());

        } catch (no.statkart.matrikkel.matrikkelapi.wsapi.v1.service.rapport.ServiceException e) {
            System.err.println("Feil, får ikke sendt matrikkelrapport til kø: " + e.getMessage());
            System.exit(-3);
        }
*/
    }
}
